<?php

namespace App\Http\Controllers;

use App\Models\Blog;
use App\Models\BlogCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class BlogController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        abort_if(auth()->user()->cannot('manage_blog'), 404);
        $blogs = Blog::all();
        return view('backend.blog.index',compact('blogs'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $blogCategories = BlogCategory::where('status',1)->orderBy('name','ASC')->get();
        return view('backend.blog.create',compact('blogCategories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = \Validator::make(
            $request->all(), [
                'title' => 'required|string|max:255',
                'category' => 'required',
            ]
        );

        if($validator->fails())
        {
            $messages = $validator->getMessageBag();
            return redirect()->back()->withInput()->with('error',$messages->first());
        }

        $check_slug = str_slug($request->title, '-');
        if (Blog::whereSlug($check_slug)->exists())
        {
            $slug = $this->incrementSlug($check_slug);
        }
        else
        {
            $slug = $check_slug;
        }

        $blog = new Blog();
        if(!empty($request->image))
        {
            $thumbnail = upload_file($request->image,'blog');
            $blog->thumbnail = $thumbnail;
        }
        $blog->title = $request->title;
        $blog->slug = $slug;
        $blog->category_id = $request->category;
        $blog->content = $request->blog_content;
        $blog->status = $request->status == 'on' ? 1 : 0;
        $blog->created_by = get_auth_data('id');
        $blog->save();

        return redirect()->route('blog.index')->with('success','Blog Created Successfully');
    }

    /**
     * Display the specified resource.
     */
    public function show(Blog $blog)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Blog $blog)
    {
        $blogCategories = BlogCategory::where('status',1)->orderBy('name','ASC')->get();
        return view('backend.blog.edit',compact('blog','blogCategories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Blog $blog)
    {
        $validator = \Validator::make(
            $request->all(), [
                'title' => 'required|string|max:255',
                'category' => 'required',
            ]
        );

        if($validator->fails())
        {
            $messages = $validator->getMessageBag();
            return redirect()->back()->withInput()->with('error',$messages->first());
        }

        $check_slug = str_slug($request->title, '-');
        if ($blog->getOriginal('slug') != $check_slug)
        {
            if (Blog::whereSlug($check_slug)->exists())
            {
                $slug = $this->incrementSlug($check_slug);
            }
            else
            {
                $slug = $check_slug;
            }
        }
        else
        {
            $slug = $check_slug;
        }

        if(!empty($request->image))
        {
            $thumbnail = upload_file($request->image,'blog',$blog->thumbnail);
            $blog->thumbnail = $thumbnail;
        }
        $blog->title = $request->title;
        $blog->slug = $slug;
        $blog->category_id = $request->category;
        $blog->content = $request->blog_content;
        $blog->status = $request->status == 'on' ? 1 : 0;
        $blog->update();

        return redirect()->route('blog.index')->with('success','Blog Updated Successfully');
    }
    
    
    public function removeblogImage(Request $request)
{
    try {
        // Validate the request to ensure 'id' is provided
        $request->validate([
            'id' => 'required',
        ]);

        // Find the TeamMember entry by ID using the TeamMembers model
        $blog = Blog::find($request->id);

        // Check if the entry exists and has an associated image
        if ($blog && $blog->thumbnail) {
            $imagePath = 'uploads/blog/' . $blog->thumbnail;

            // Use a helper function to delete the image file
            delete_file('blog',$blog->thumbnail);

            // Remove the image path from the database
            $blog->thumbnail = null;
            $blog->save();

            return response()->json(['success' => true, 'message' => 'Image removed successfully.']);
        } else {
            return response()->json(['success' => false, 'message' => 'Team member or image not found.']);
        }
    } catch (\Exception $e) {
        // Log any exceptions that occur
        Log::error('Error removing image: ' . $e->getMessage());
        return response()->json(['success' => false, 'message' => 'An error occurred while removing the image.']);
    }
}
    

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Blog $blog)
    {
        abort_if(auth()->user()->cannot('delete_blog'), 404);
        delete_file('blog',$blog->thumbnail);
        $blog->delete();
        return redirect()->back()->with('success','Blog Deleted Successfully');
    }

    public function updateStatus(Request $request)
    {
        abort_if(auth()->user()->cannot('edit_blog'), 404);
        $blog = Blog::find($request->id);
        $blog->status = $request->status;
        $blog->update();
        return response()->json([
            'status' => 'success',
            'message' => __('Status updated')
        ]);
    }

    public function incrementSlug($slug)
    {
        for($i = 1; $i <= 100; $i++)
        {
            $new_slug = $slug . '-' . $i;
            if(!Blog::whereSlug($new_slug)->exists())
            {
                return $new_slug;
            }
        }
    }
    
    public function uploader(Request $request)
{
    if ($request->hasFile('upload')) {
        $file = $request->file('upload');
        $name = time() . rand(1, 100) . '.' . $file->getClientOriginalExtension();
        $upload_path = 'assets/media/image/';
        $file->move($upload_path, $name);

        $url = asset($upload_path . $name);
        // return response()->json([
        //     'url' => $url
        // ]);
        return response()->json(['fileName' => $name, 'uploaded'=> 1, 'url' => $url]);
    }
    return response()->json(['message' => 'No file uploaded'], 400);
}

}
