<?php

namespace App\Http\Controllers;

use App\Models\Pages;
use App\Models\PageSection;
use Illuminate\Http\Request;
use DB;

class PagesController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        abort_if(auth()->user()->cannot('manage_pages'), 404);
        $pages = Pages::all();
        return view('backend.pages.index',compact('pages'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        abort_if(auth()->user()->cannot('create_pages'), 404);
        $pageSections = PageSection::where('status',1)->get();
        $career = DB::table('career')->get();
        $faq_categories = DB::table('faq_categories')->where('status','1')->get();
        return view('backend.pages.create',compact('pageSections','career','faq_categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // return $request->all();
        abort_if(auth()->user()->cannot('create_pages'), 404);
        $validator = \Validator::make(
            $request->all(), [
                'name' => 'required|string|max:255',
            ]
        );

        if($validator->fails())
        {
            $messages = $validator->getMessageBag();
            return redirect()->back()->withInput()->with('error',$messages->first());
        }

        $check_slug = str_slug($request->name, '-');
        if (Pages::whereSlug($check_slug)->exists())
        {
            $slug = $this->incrementSlug($check_slug);
        }
        else
        {
            $slug = $check_slug;
        }

        $page = new Pages();
        if(!empty($request->banner_image))
        {
            $banner_image = upload_file($request->banner_image,'page-banner');
            $page->banner_image = $banner_image;
        }
        if(!empty($request->mobile_banner_image))
        {
            $mobile_banner_image = upload_file($request->mobile_banner_image,'page-banner');
            $page->mobile_banner_image = $mobile_banner_image;
        }
        $page->name = $request->name;
        $page->description = $request->description;
        $page->metatitle = $request->metatitle;
        $page->metakeyword = $request->metakeyword;
        $page->metadescription = $request->metadescription;
        $page->team_members = $request->team_members ?? 0;
        $page->partners = $request->partners ?? 0;
        $page->our_journey = $request->our_journey ?? 0;
        $page->career = $request->career;
        // $page->faq_categories = $request->faq_categories;
       
        $page->slug = $slug;
        
        $sections = [];
        $faq_categories = [];
        foreach($request->sections as $key => $section){
            $sections[] = $section;
            $faq_categories[] = $request->faq_categories[$key] == null ? null : $request->faq_categories[$key];
        }
        $page->sections = json_encode($sections);
         $page->faq_categories = json_encode($faq_categories);
        $page->status = $request->status == 'on' ? 1 : 0;
        $page->save();
        
        if (!empty($request->pdfs)) {
        foreach ($request->file('pdfs') as $pdfFile) {
                // Assuming upload_file is a custom function to handle file uploads
                $pdfPath = upload_file($pdfFile, 'page-pdf');
                
                DB::table('pages_pdf')->insert([
                    'page_id' => $page->id,
                    'pdf_name' => $pdfPath,  // Assuming 'column_name2' stores the file path or name
                ]);
            }
        }

        return redirect()->route('pages.index')->with('success','Page Created Successfully');
    }

    /**
     * Display the specified resource.
     */
    public function show(Pages $pages)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Pages $page)
    {
        abort_if(auth()->user()->cannot('edit_pages'), 404);
        $pageSections = PageSection::where('status',1)->get();
        $career = DB::table('career')->get();
        $faq_categories = DB::table('faq_categories')->where('status','1')->get();
        $pdfs = DB::table('pages_pdf')->where('page_id',$page->id)->get();
        return view('backend.pages.edit',compact('page','pageSections','career','faq_categories','pdfs'));
    }
    
    public function pdfDelete($id) {
    // Retrieve the record
    $pages_pdf = DB::table('pages_pdf')->where('id', $id)->first();
    
    if ($pages_pdf) {
        // Delete the file (assuming 'pdf_name' is the correct column name)
        delete_file('page-pdf', $pages_pdf->pdf_name);
        
        // Delete the record from the database
        DB::table('pages_pdf')->where('id', $id)->delete();
    }
    
    // Redirect back to the previous page
    return redirect()->back();
}


    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Pages $page)
    {
        abort_if(auth()->user()->cannot('edit_pages'), 404);
        $validator = \Validator::make(
            $request->all(), [
                'name' => 'required|string|max:255',
            ]
        );

        if($validator->fails())
        {
            $messages = $validator->getMessageBag();
            return redirect()->back()->withInput()->with('error',$messages->first());
        }

        $check_slug = str_slug($request->name, '-');
        if ($page->getOriginal('slug') != $check_slug)
        {
            if (Pages::whereSlug($check_slug)->exists())
            {
                $slug = $this->incrementSlug($check_slug);
            }
            else
            {
                $slug = $check_slug;
            }
        }
        else
        {
            $slug = $check_slug;
        }

        if(!empty($request->banner_image))
        {
            $banner_image = upload_file($request->banner_image,'page-banner',$page->banner_image);
            $page->banner_image = $banner_image;
        }
        if(!empty($request->mobile_banner_image))
        {
            $mobile_banner_image = upload_file($request->mobile_banner_image,'page-banner');
            $page->mobile_banner_image = $mobile_banner_image;
        }
        $page->name = $request->name;
        $page->description = $request->description;
        $page->metatitle = $request->metatitle;
        $page->metakeyword = $request->metakeyword;
        $page->metadescription = $request->metadescription;
        $page->team_members = $request->team_members ?? 0;
        $page->partners = $request->partners ?? 0;
        $page->our_journey = $request->our_journey ?? 0;
        $page->career = $request->career;
        $page->faq_categories = $request->faq_categories;
        $page->slug = $slug;
        $page->sections = json_encode($request->sections);
        $page->status = $request->status == 'on' ? 1 : 0;
        $page->update();
        
        
        if (!empty($request->pdfs)) {
        foreach ($request->file('pdfs') as $pdfFile) {
                $pdfPath = upload_file($pdfFile, 'page-pdf');
                
                DB::table('pages_pdf')->insert([
                    'page_id' => $page->id,
                    'pdf_name' => $pdfPath, 
                ]);
            }
        }

        return redirect()->route('pages.index')->with('success','Page Updated Successfully');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Pages $page)
    {
        abort_if(auth()->user()->cannot('delete_pages'), 404);
        delete_file('page-banner',$page->banner_image);
        $page->delete();
        return redirect()->back()->with('success','Page Deleted Successfully');
    }
    
     public function removeImage(Request $request)
{
    try {
        // Validate the request to ensure 'id' is provided
        $request->validate([
            'id' => 'required',
        ]);

        // Find the HeroSection entry by ID
        $page = Pages::find($request->id);

        // Check if the entry exists and has an associated image
        if ($page && $page->banner_image) {
            $imagePath = 'uploads/global-reach/' . $page->banner_image;

            delete_file('page-banner',$page->banner_image);

            // Remove the image path from the database
            $page->image = null;
            $page->save();

            return response()->json(['success' => true, 'message' => 'Image removed successfully.']);
        } else {
            return response()->json(['success' => false, 'message' => 'Hero section or image not found.']);
        }
    } catch (\Exception $e) {
        // Log any exceptions that occur
        Log::error('Error removing image: ' . $e->getMessage());
        return response()->json(['success' => false, 'message' => 'An error occurred while removing the image.']);
    }
}

    public function updateStatus(Request $request)
    {
        abort_if(auth()->user()->cannot('edit_pages'), 404);
        $pages = Pages::find($request->id);
        $pages->status = $request->status;
        $pages->update();
        return response()->json([
            'status' => 'success',
            'message' => __('Status updated')
        ]);
    }

    public function incrementSlug($slug)
    {
        for($i = 1; $i <= 100; $i++)
        {
            $new_slug = $slug . '-' . $i;
            if(!Pages::whereSlug($new_slug)->exists())
            {
                return $new_slug;
            }
        }
    }
}
