<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class RoleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $authUser = auth()->user();
        abort_if($authUser->cannot('manage_role'), 404);
        $roles = Role::with(['permissions', 'users'])
            ->where('name','!=','Superadmin')
            ->get();
        return view('backend.role.index',compact('roles'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $authUser = auth()->user();
        abort_if($authUser->cannot('create_role'), 404);
        $permissions = Permission::orderBy('group')->get()->groupBy(function($data) {
            return $data->group;
        });
        return view('backend.role.create',compact('permissions'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $authUser = auth()->user();
        abort_if($authUser->cannot('create_role'), 404);
        $validator = \Validator::make(
            $request->all(), [
                'name' => 'required|unique:roles',
                'permissions' => 'required',
            ]
        );
        if($validator->fails())
        {
            $messages = $validator->getMessageBag();
            return redirect()->back()->with('error', $messages->first());
        }
        $role = new Role();

        $role->name = $request->name;
        $role->save();

        foreach($request->permissions as $permission_id){
            $permission = Permission::find($permission_id);
            $role->givePermissionTo($permission);
        }
        return redirect()->route('role.index')->with('success', __('Role created successfully'));
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $authUser = auth()->user();
        $role = Role::where('id', $id)->with(['permissions'])->first();
        abort_if($authUser->cannot('edit_role'), 404);
        $permissions = Permission::orderBy('group')->get()->groupBy(function($data) {
            return $data->group;
        });
        $selected_permissions = $role->permissions->pluck('id')->toArray();
        return view('backend.role.edit', compact('permissions', 'selected_permissions', 'role'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {   return $request->all();
        $authUser = auth()->user();
        abort_if($authUser->cannot('edit_role'), 404);
        $role = Role::find($id);
        $validator = \Validator::make(
            $request->all(), [
                'name' => 'required|unique:roles,name,' . $role->id . ',id',
                'permissions' => 'required',
            ]
        );
        if ($validator->fails()) {
            $messages = $validator->getMessageBag();
            return redirect()->back()->withInput()->with('error', $messages->first());
        }
        $permissions = Permission::all();
        foreach ($permissions as $permission) {
            $role->revokePermissionTo($permission);
        }
        foreach ($request->permissions as $permission_id) {
            $permission = Permission::find($permission_id);
            $role->givePermissionTo($permission);
        }
        $role->name = $request->name;
        $role->update();
        return redirect()->route('role.index')->with('success', __('Role updated successfully'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $authUser = auth()->user();
        abort_if($authUser->cannot('delete_role'), 404);
        $role = Role::where('id', $id)->with(['users', 'permissions'])->first();
        if ($role->users->count() > 0) {
            return redirect()->back()->withInput()->with('error', __('Role is assign to users'));
        } else {
            foreach ($role->permissions as $permission) {
                $role->revokePermissionTo($permission);
            }
            $role->delete();
            return redirect()->back()->with('success', __('Role deleted successfully'));
        }
    }
}
