<?php

namespace App\Http\Controllers;

use App\Mail\TestMail;
use App\Models\Settings;
use App\Models\Timezones;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use App\Models\MenuItem;
use Illuminate\Support\Str;
use App\Models\Pages;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class SettingsController extends Controller
{
    public function index(Request $request)
    {
        abort_if(auth()->user()->cannot('manage_site_settings'), 404);
        
        if($request->isMethod('POST'))
        {
            $validator = \Validator::make(
                $request->all(), [
                    'app_name' => 'required',
                    'favicon' => 'image|mimes:png',
                    'logo' => 'image|mimes:png',
                    'logo_light' => 'image|mimes:png',
                ]
            );
            if ($validator->fails()) {
                $messages = $validator->getMessageBag();
                return redirect()->back()->withInput()->with('error', $messages->first());
            }


            $timezone = ['APP_TIMEZONE' => $request->timezone];

            setEnv($timezone);

            foreach($request->only(['app_name','footer_text_admin','logo_light','favicon','logo']) as $key => $value)
            {
                if ($key == 'logo_light') {
                    $logo_light = 'logo_light.png';
                    $request->logo_light->move(public_path('assets/media/img'), $logo_light);
                    $value = 'logo_light.png';
                }
                
                if ($key == 'favicon') {
                    $favicon = 'favicon.png';
                    $request->favicon->move(public_path('assets/media/img'), $favicon);
                    $value = 'favicon.png';
                }
                
                if ($key == 'logo') {
                    $logo = 'logo.png';
                    $request->logo->move(public_path('assets/media/img'), $logo);
                    $value = 'logo.png';
                }
                
                
                $save = ['key' => $key, 'value' => $value];
                Settings::updateOrCreate(
                    [
                        'key' => $key
                    ], $save
                );
            }

            return redirect()->back()->with('success', __('Settings updated'));
        }

        $timezones = Timezones::all();
        $settings = get_all_setting();
        return view('backend.settings.index',compact('settings','timezones'));
    }

    public function smtp(Request $request)
    {
        abort_if(auth()->user()->cannot('manage_site_settings'), 404);
        if($request->isMethod('POST'))
        {
            $enable_email = isset($request->enable_email) && $request->enable_email == 'on' ? 1 : 0;

            if($enable_email)
            {
                $validator = \Validator::make(
                    $request->all(), [
                        'mail_driver' => 'required|string|max:255',
                        'mail_host' => 'required|string|max:255',
                        'mail_port' => 'required|string|max:255',
                        'mail_username' => 'required|string|max:255',
                        'mail_password' => 'required|string|max:255',
                        'mail_encryption' => 'required|string|max:255',
                        'mail_from_address' => 'required|string|max:255',
                        'mail_from_name' => 'required|string|max:255',
                    ]
                );
                if ($validator->fails())
                {
                    $messages = $validator->getMessageBag();
                    return redirect()->back()->withInput()->with('error', $messages->first());
                }

                $array = [
                    'MAIL_DRIVER' => $request->mail_driver,
                    'MAIL_HOST' => $request->mail_host,
                    'MAIL_PORT' => $request->mail_port,
                    'MAIL_USERNAME' => $request->mail_username,
                    'MAIL_PASSWORD' => $request->mail_password,
                    'MAIL_ENCRYPTION' => $request->mail_encryption,
                    'MAIL_FROM_NAME' => $request->mail_from_name,
                    'MAIL_FROM_ADDRESS' => $request->mail_from_address,
                ];
                setEnv($array);
            }

            Settings::updateOrCreate(
                [
                    'key' => 'enable_email'
                ], ['key' => 'enable_email', 'value' => $enable_email]
            );

            return redirect()->back()->withInput()->with('success', 'SMTP settings updated');
        }

        $settings = get_all_setting();
        return view('backend.settings.smtp',compact('settings'));
    }

    public function testMail(Request $request)
    {
        if($request->isMethod('POST'))
        {
            try
            {
                Mail::to($request->email)->send(new TestMail());
                return redirect()->back()->with('success', __('Mail sent'));
            }
            catch (\Exception $e)
            {
                return redirect()->back()->withInput()->with('error', $e->getMessage());
            }
        }
        return view('backend.settings.test-mail');
    }

    public function footer(Request $request)
    {
       // abort_if(auth()->user()->cannot('social_media_settings'), 404);
        if($request->isMethod('POST'))
        {   
            foreach($request->except(['_token']) as $key => $value)
            {
                if($key == 'company'){
                    $value = json_encode($value);
                }
                if($key == 'quick_links'){
                    $value = json_encode($value);
                }
                if($key == 'support'){
                    $value = json_encode($value);
                }
                
                if ($key == 'footer_logo') {
                    $footer_logo = 'footer_logo.png';
                    $request->footer_logo->move(public_path('assets/media/img'), $footer_logo);
                    $value = 'footer_logo.png';
                }
                
                $save = ['key' => $key, 'value' => $value];
                Settings::updateOrCreate(
                    [
                        'key' => $key
                    ], $save
                );
            }

            return redirect()->back()->with('success', __('Settings updated'));
        }
        $settings = get_all_setting();
        $pages = Pages::all();
        $menuItems = MenuItem::orderBy('order')->get();
        return view('backend.settings.footer',compact('settings','pages','menuItems'));
    }
    
    public function menuindex()
    {
        $menuItems = MenuItem::whereNull('parent_id')->with('children')->orderBy('order')->get();
        return view('backend.settings.menu.index', compact('menuItems'));
    }
    
    public function menuCreate()
    {
        $menuItems = MenuItem::all();
        return view('backend.settings.menu.create', compact('menuItems'));
    }
    
    public function menuStore(Request $request)
    {
          // Validate the request data
        $request->validate([
            'title' => 'required|string|max:255',
            'url' => 'required',
            'parent_id' => 'nullable|exists:menu_items,id'
        ]);
    
        // Get all input data
        $input = $request->all();
    
        // Create a slug from the title
        $input['slug'] = Str::slug($request->title);
        if(!empty($request->image))
        {
            $image = upload_file($request->image,'manu-section');
            $input['image'] = $image;
        }
        MenuItem::create($input);
        return redirect()->route('settings.menu');
    }
    
    public function menuEdit($id)
    {
        $menuItem = MenuItem::findOrFail($id);
        $menuItems = MenuItem::all();
        return view('backend.settings.menu.edit', compact('menuItem', 'menuItems'));
    }

    public function menuUpdate(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'url' => 'required',
            'parent_id' => 'nullable|exists:menu_items,id'
        ]);
    
        // Get all input data
        $input = $request->all();
    
        // Create a slug from the title
        $input['slug'] = Str::slug($request->title);
        
        $menuItem = MenuItem::findOrFail($id);
        if(!empty($request->image))
        {
            $image = upload_file($request->image,'manu-section',$menuItem->image);
            $input['image'] = $image;
        }
        $menuItem->update($input);
        return redirect()->route('settings.menu');
    }
    
    public function menuDelete($id)
    {
        $menuItem = MenuItem::findOrFail($id);
        $menuItem->delete();
        return redirect()->route('settings.menu');
    }
}
