<?php

namespace App\Http\Controllers;

use App\Models\TeamMembers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class TeamMembersController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        abort_if(auth()->user()->cannot('manage_team_members'), 404);
        $teamMembers = TeamMembers::where('type','team-member')->get();
        return view('backend.team-members.index',compact('teamMembers'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        abort_if(auth()->user()->cannot('create_team_members'), 404);
        return view('backend.team-members.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        abort_if(auth()->user()->cannot('create_team_members'), 404);
        $validator = \Validator::make(
            $request->all(), [
                'name' => 'required|string|max:255',
                'designation' => 'required|string|max:255',
            ]
        );

        if($validator->fails())
        {
            $messages = $validator->getMessageBag();
            return redirect()->back()->withInput()->with('error',$messages->first());
        }


        $teamMember = new TeamMembers();
        if(!empty($request->image))
        {
            $image = upload_file($request->image,'team-members');
            $teamMember->image = $image;
        }
        $teamMember->name = $request->name;
        $teamMember->type = 'team-member';
        $teamMember->designation = $request->designation;
        $teamMember->department = $request->department;
        $teamMember->description = $request->description;
        $teamMember->status = $request->status == 'on' ? 1 : 0;
        $teamMember->save();

        return redirect()->route('team-member.index')->with('success','Team Member Added Successfully');
    }

    /**
     * Display the specified resource.
     */
    public function show(TeamMembers $teamMembers)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(TeamMembers $teamMember)
    {
        abort_if(auth()->user()->cannot('edit_team_members'), 404);
        return view('backend.team-members.edit',compact('teamMember'));
    }



public function removeImage(Request $request)
{
    try {
        // Validate the request to ensure 'id' is provided
        $request->validate([
            'id' => 'required',
        ]);

        // Find the TeamMember entry by ID using the TeamMembers model
        $teamMember = TeamMembers::find($request->id);

        // Check if the entry exists and has an associated image
        if ($teamMember && $teamMember->image) {
            $imagePath = 'uploads/team-members/' . $teamMember->image;

            // Use a helper function to delete the image file
            delete_file('team-members', $teamMember->image);

            // Remove the image path from the database
            $teamMember->image = null;
            $teamMember->save();

            return response()->json(['success' => true, 'message' => 'Image removed successfully.']);
        } else {
            return response()->json(['success' => false, 'message' => 'Team member or image not found.']);
        }
    } catch (\Exception $e) {
        // Log any exceptions that occur
        Log::error('Error removing image: ' . $e->getMessage());
        return response()->json(['success' => false, 'message' => 'An error occurred while removing the image.']);
    }
}


    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, TeamMembers $teamMember)
    {
        abort_if(auth()->user()->cannot('edit_team_members'), 404);
        $validator = \Validator::make(
            $request->all(), [
                'name' => 'required|string|max:255',
                'designation' => 'required|string|max:255',
            ]
        );

        if($validator->fails())
        {
            $messages = $validator->getMessageBag();
            return redirect()->back()->withInput()->with('error',$messages->first());
        }


        if(!empty($request->image))
        {
            $image = upload_file($request->image,'team-members',$teamMember->image);
            $teamMember->image = $image;
        }
        $teamMember->name = $request->name;
        $teamMember->designation = $request->designation;
        $teamMember->department = $request->department;
        $teamMember->description = $request->description;
        $teamMember->status = $request->status == 'on' ? 1 : 0;
        $teamMember->update();

        return redirect()->route('team-member.index')->with('success','Team Member Updated Successfully');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(TeamMembers $teamMember)
    {
        abort_if(auth()->user()->cannot('delete_team_members'), 404);
        delete_file('team-members',$teamMember->image);
        $teamMember->delete();
        return redirect()->back()->with('success','Team Member Deleted Successfully');
    }

    public function updateStatus(Request $request)
    {
        abort_if(auth()->user()->cannot('edit_team_members'), 404);
        $pages = TeamMembers::find($request->id);
        $pages->status = $request->status;
        $pages->update();
        return response()->json([
            'status' => 'success',
            'message' => __('Status updated')
        ]);
    }
}
