<?php

namespace App\Http\Controllers;

use App\Models\BlogCategory;
use Illuminate\Http\Request;

class BlogCategoryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        abort_if(auth()->user()->cannot('manage_blog_category'), 404);
        $blogCategories = BlogCategory::all();
        return view('backend.blog-category.index',compact('blogCategories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        abort_if(auth()->user()->cannot('create_blog_category'), 404);
        return view('backend.blog-category.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        abort_if(auth()->user()->cannot('create_blog_category'), 404);
        $validator = \Validator::make(
            $request->all(), [
                'name' => 'required|unique:blog_categories',
            ]
        );
        if($validator->fails())
        {
            $messages = $validator->getMessageBag();
            return redirect()->back()->with('error', $messages->first());
        }

        $check_slug = str_slug($request->name, '-');
        if (BlogCategory::whereSlug($check_slug)->exists()) {
            $slug = $this->incrementSlug($check_slug);
        } else {
            $slug = $check_slug;
        }

        $category = new BlogCategory();
        $category->name = $request->name;
        $category->description = $request->description;
        $category->slug = $slug;
        $category->status = $request->status == 'on' ? 1 : 0;
        $category->save();
        return redirect()->back()->with('success', __('Category created'));
    }

    /**
     * Display the specified resource.
     */
    public function show(BlogCategory $blogCategory)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(BlogCategory $blogCategory)
    {
        abort_if(auth()->user()->cannot('edit_blog_category'), 404);
        return view('backend.blog-category.edit',compact('blogCategory'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, BlogCategory $blogCategory)
    {
        abort_if(auth()->user()->cannot('edit_blog_category'), 404);
        $validator = \Validator::make(
            $request->all(), [
                'name' => 'required|unique:blog_categories,name,' . $blogCategory->id,
            ]
        );
        if($validator->fails())
        {
            $messages = $validator->getMessageBag();
            return redirect()->back()->with('error', $messages->first());
        }

        $check_slug = str_slug($request->name, '-');
        if ($blogCategory->getOriginal('slug') != $check_slug) {
            if (BlogCategory::whereSlug($check_slug)->exists()) {
                $slug = $this->incrementSlug($check_slug);
            } else {
                $slug = $check_slug;
            }
        } else {
            $slug = $check_slug;
        }

        $blogCategory->name = $request->name;
        $blogCategory->description = $request->description;
        $blogCategory->slug = $slug;
        $blogCategory->status = $request->status == 'on' ? 1 : 0;
        $blogCategory->update();
        return redirect()->back()->with('success', __('Category updated'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(BlogCategory $blogCategory)
    {
        abort_if(auth()->user()->cannot('delete_blog_category'), 404);
        $blogCategory->delete();
        return redirect()->back()->with('success', __('Category deleted'));
    }

    public function updateStatus(Request $request)
    {
        abort_if(auth()->user()->cannot('edit_blog_category'), 404);
        $blogCategory = BlogCategory::find($request->id);
        $blogCategory->status = $request->status;
        $blogCategory->update();
        return response()->json([
            'status' => 'success',
            'message' => __('Status updated')
        ]);
    }

    public function incrementSlug($slug)
    {
        for($i = 1; $i <= 100; $i++)
        {
            $new_slug = $slug . '-' . $i;
            if(!BlogCategory::whereSlug($new_slug)->exists())
            {
                return $new_slug;
            }
        }
    }
}
