<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class UsersController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        abort_if(auth()->user()->cannot('manage_staff'), 404);
        $users = User::where('type','Staff')->where('created_by',auth()->id())->get();
        return view('backend.users.index',compact('users'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        abort_if(auth()->user()->cannot('create_staff'), 404);
        $roles = Role::where('name','!=','Superadmin')->get();
        return view('backend.users.create',compact('roles'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {   
        abort_if(auth()->user()->cannot('create_staff'), 404);
        $validator = \Validator::make(
            $request->all(), [
                'name' => 'required|string|max:255',
                'email' => 'required|string|email|max:255|unique:users',
                'password' => 'required|string|min:6|confirmed',
                'role' => 'required',
                // 'image' => [
                //     'mimes:jpeg,png,jpg|max:20480',
                // ],
            ]
        );

        if($validator->fails())
        {
            $messages = $validator->getMessageBag();
            return redirect()->back()->with('error',$messages->first());
        }

        $user = new User();

        if(!empty($request->image))
        {
            $profile = upload_file($request->image,'profile');
            $user->profile = $profile;
        }

        $user->name = $request->name;
        $user->email = $request->email;
        $user->phone_no = $request->phone_no;
        $user->password = Hash::make($request->password);
        $user->created_by = auth()->id();
        $user->status = 1;
        $user->type = 'Staff';
        $user->save();

        $role = Role::find($request->role);
        $user->assignRole($role);

        return redirect()->route('users.index')->with('success','User created successfully');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        abort_if(auth()->user()->cannot('edit_staff'), 404);
        $user = User::find($id);
        $roles = Role::where('name','!=','Superadmin')->get();
        return view('backend.users.edit',compact('user','roles'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        
        abort_if(auth()->user()->cannot('edit_staff'), 404);
        $validator = \Validator::make(
            $request->all(), [
                'name' => 'required|string|max:255',
                'email' => 'required|string|email|max:255',
                'role' => 'required',
                // 'image' => [
                //     'mimes:jpeg,png,jpg|max:20480',
                // ],
            ]
        );

        if($validator->fails())
        {
            $messages = $validator->getMessageBag();
            return redirect()->back()->with('error',$messages->first());
        }

        $user = User::find($id);

        if(!empty($request->image))
        {
            $profile = upload_file($request->image,'profile',$user->profile);
            $user->profile = $profile;
        }

        $user->name = $request->name;
        $user->email = $request->email;
        $user->phone_no = $request->phone_no;
        $user->update();

        if($user->roles()->first()->id != $request->role)
        {
            $user->removeRole($user->roles()->first()->id);
            $user->assignRole($request->role);
        }

        return redirect()->route('users.index')->with('success','User updated successfully');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        abort_if(auth()->user()->cannot('delete_staff'), 404);
        $user = User::find($id);
        if($user->roles()->count() > 0) {
            $user->removeRole($user->roles()->first());
        }
        delete_file('profile', $user->profile);
        $user->delete();
        return redirect()->back()->withInput()->with('success',__('User deleted'));
    }

    public function updateStatus(Request $request)
    {
        $user = User::find($request->id);
        $user->status = $request->status;
        $user->update();
        return response()->json([
            'status' => 'success',
            'message' => __('Status updated')
        ]);
    }

    public function resetPassword(Request $request,$user_id)
    {
        if($request->isMethod('POST'))
        {
            $validator = \Validator::make(
                $request->all(), [
                    'password' => 'required|min:6',
                    'confirm_password' => 'required|same:password',
                ]
            );
            if ($validator->fails()) {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }
            $user = User::find($user_id);

            $user->password = Hash::make($request->password);
            $user->update();
            return redirect()->back()->with('success',__('Password changed successfully'));
        }

        return view('backend.users.reset-password', compact('user_id'));
    }

    public function profile()
    {
        $user = auth()->user();
        return view('backend.users.profile', compact('user'));
    }

    public function updateProfile(Request $request)
    {
        $user = auth()->user();

        $validationParam = [
            'name' => 'required|string|max:255',
            // 'image' => [
            //     'mimes:jpeg,png,jpg|max:20480',
            // ]
        ];

        $validator = \Validator::make(
            $request->all(), $validationParam
        );

        if($validator->fails())
        {
            $messages = $validator->getMessageBag();
            return redirect()->back()->with('error',$messages->first());
        }

        if(!empty($request->image))
        {
            $profile = upload_file($request->image,'profile',$user->profile);
            $user->profile = $profile;
        }

        $user->name = $request->name;
        $user->phone_no = $request->phone_no;
        $user->update();

        return redirect()->back()->with('success','Profile updated successfully');
    }

    public function updateEmail(Request $request)
    {
        $user = auth()->user();

        $validationParam = [
            'email' => 'required|email|unique:users,email,' . $user->id,
            'password' => 'required|string|max:255',
        ];

        $validator = \Validator::make(
            $request->all(), $validationParam
        );

        if($validator->fails())
        {
            $messages = $validator->getMessageBag();
            return redirect()->back()->with('error',$messages->first());
        }
        if(Hash::check($request->password, $user->password))
        {
            $user->email = $request->email;
            $user->update();
        }
        else
        {
            return redirect()->back()->with('error',__('Password does not match'));
        }

        return redirect()->back()->with('success','Email updated successfully');
    }
    
    
    // Remove Imagee
    
    public function removeImage(Request $request)
    {
        // return "ok";
        try {
            
            $request->validate([
                'user_id' => 'required',
            ]);

            $user = User::find($request->user_id);

           if ($user && $user->profile) {
            $imagePath = 'uploads/profile/' . $user->profile;

            // Use a helper function to delete the image file
            delete_file('profile',$user->profile);

            // Remove the image path from the database
            $user->profile = null;
            $user->save();

                return response()->json(['success' => true, 'message' => 'Image removed successfully.']);
            } else {
                return response()->json(['success' => false, 'message' => 'User or image not found.']);
            }
        } catch (\Exception $e) {
            Log::error('Error removing image: ' . $e->getMessage());
            return response()->json(['success' => false, 'message' => 'An error occurred while removing the image.']);
        }
    }
    
    

    public function changePassword(Request $request)
    {
        if(auth()->check())
        {
            $validator = \Validator::make(
                $request->all(), [
                    'current_password' => 'required',
                    'password' => 'required|min:6',
                    'confirm_password' => 'required|same:password',
                ]
            );
            if ($validator->fails()) {
                $messages = $validator->getMessageBag();
                return redirect()->back()->with('error', $messages->first());
            }
            $user = User::find(auth()->user()->id);
            if(Hash::check($request->current_password, $user->password))
            {
                $user->password = Hash::make($request->password);
                $user->update();
                return redirect()->back()->with('success',__('Password changed successfully'));
            }
            else
            {
                return redirect()->back()->with('error',__('Current password does not match'));
            }
        }else
        {
            return redirect()->route('login');
        }
    }
}
