<?php

namespace TJCore;

use Elementor\Controls_Manager;
use Elementor\Widget_Base;

if (! defined('ABSPATH')) {
  exit; // Exit if accessed directly
}

/**
 * Elementor Widget: TJ Project Filter
 */
class TJ_Project_Filter extends Widget_Base
{

  public function get_name()
  {
    return 'tj-project-filter';
  }

  public function get_title()
  {
    return __('TJ Project Filter', 'tjcore');
  }

  public function get_icon()
  {
    return 'eicon-post tj-icon';
  }

  public function get_categories()
  {
    return ['tjcore'];
  }

  public function get_script_depends()
  {
    return ['tjcore-script'];
  }

  public function get_style_depends()
  {
    return ['tjcore-style'];
  }

  public function get_keywords()
  {
    return ['project', 'projects', 'tj project', 'tj', 'tj addons', 'tjcore'];
  }

  public function get_custom_help_url()
  {
    return 'https://go.elementor.com/';
  }

  /**
   * Register widget controls
   */
  protected function register_controls()
  {

    $taxonomy = 'projects-cat';

    // Project Query Section
    $this->start_controls_section(
      'tj_project_query',
      [
        'label' => esc_html__('Project Query', 'tjcore'),
      ]
    );

    $this->add_control(
      'posts_per_page',
      [
        'label'       => esc_html__('Projects Per Page', 'tjcore'),
        'description' => esc_html__('Leave blank or enter -1 for all.', 'tjcore'),
        'type'        => Controls_Manager::NUMBER,
        'default'     => 6,
      ]
    );

    $this->add_control(
      'category',
      [
        'label'       => esc_html__('Include Categories', 'tjcore'),
        'description' => esc_html__('Select categories to include or leave blank for all.', 'tjcore'),
        'type'        => Controls_Manager::SELECT2,
        'multiple'    => true,
        'options'     => tj_get_categories($taxonomy),
        'label_block' => true,
      ]
    );

    $this->add_control(
      'tj_post_title_word',
      [
        'label'       => esc_html__('Title Word Limit', 'tjcore'),
        'description' => esc_html__('Set how many words you want to display.', 'tjcore'),
        'type'        => Controls_Manager::NUMBER,
        'default'     => 6,
      ]
    );

    $this->add_control(
      'tj_category',
      [
        'label'        => esc_html__('Enable Category?', 'tjcore'),
        'type'         => Controls_Manager::SWITCHER,
        'label_on'     => esc_html__('Yes', 'tjcore'),
        'label_off'    => esc_html__('No', 'tjcore'),
        'return_value' => 'yes',
        'default'      => 'yes',
        'separator'    => 'before',
      ]
    );

    $this->add_control(
      'tj_all_btn',
      [
        'label'       => esc_html__('Filter Button', 'tjcore'),
        'label_block' => true,
        'type'        => Controls_Manager::TEXT,
        'default'     => esc_html__('All', 'tjcore'),
      ]
    );

    $this->end_controls_section();

    $this->start_controls_section(
      'section_column_settings',
      [
        'label' => esc_html__('Column Settings', 'tjcore'),
        'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
      ]
    );

    $this->add_control(
      'desktop_column',
      [
        'label' => esc_html__('Columns for Desktop', 'tjcore'),
        'description' => esc_html__('Screen width equal to or greater than 1200px', 'tjcore'),
        'type' => \Elementor\Controls_Manager::SELECT,
        'options' => [
          12 => esc_html__('1 Columns', 'tjcore'),
          6 => esc_html__('2 Columns', 'tjcore'),
          4 => esc_html__('3 Columns', 'tjcore'),
          3 => esc_html__('4 Columns', 'tjcore'),
          2 => esc_html__('6 Columns', 'tjcore'),
        ],
        'separator' => 'before',
        'default' => '4',
        'style_transfer' => true,
      ]
    );

    $this->add_control(
      'laptop_column',
      [
        'label' => esc_html__('Columns for Laptop', 'tjcore'),
        'description' => esc_html__('Screen width equal to or greater than 992px', 'tjcore'),
        'type' => \Elementor\Controls_Manager::SELECT,
        'options' => [
          12 => esc_html__('1 Columns', 'tjcore'),
          6 => esc_html__('2 Columns', 'tjcore'),
          4 => esc_html__('3 Columns', 'tjcore'),
          3 => esc_html__('4 Columns', 'tjcore'),
          2 => esc_html__('6 Columns', 'tjcore'),
        ],
        'separator' => 'before',
        'default' => '6',
        'style_transfer' => true,
      ]
    );

    $this->add_control(
      'tablet_column',
      [
        'label' => esc_html__('Columns for Tablet', 'tjcore'),
        'description' => esc_html__('Screen width equal to or greater than 768px', 'tjcore'),
        'type' => \Elementor\Controls_Manager::SELECT,
        'options' => [
          12 => esc_html__('1 Columns', 'tjcore'),
          6 => esc_html__('2 Columns', 'tjcore'),
          4 => esc_html__('3 Columns', 'tjcore'),
          3 => esc_html__('4 Columns', 'tjcore'),
          2 => esc_html__('6 Columns', 'tjcore'),
        ],
        'separator' => 'before',
        'default' => '6',
        'style_transfer' => true,
      ]
    );
    $this->add_control(
      'mobile_column',
      [
        'label' => esc_html__('Columns for Mobile', 'tjcore'),
        'description' => esc_html__('Screen width less than 768px', 'tjcore'),
        'type' => \Elementor\Controls_Manager::SELECT,
        'options' => [
          12 => esc_html__('1 Columns', 'tjcore'),
          6 => esc_html__('2 Columns', 'tjcore'),
          4 => esc_html__('3 Columns', 'tjcore'),
          3 => esc_html__('4 Columns', 'tjcore'),
          2 => esc_html__('6 Columns', 'tjcore'),
        ],
        'separator' => 'before',
        'default' => '12',
        'style_transfer' => true,
      ]
    );

    $this->end_controls_section();


    /**
     * Filter Button Style Controls
     */
    $this->start_controls_section(
      'tj_filter_button_style',
      [
        'label' => esc_html__('Filter Buttons', 'tjcore'),
        'tab'   => Controls_Manager::TAB_STYLE,
      ]
    );

    // Typography
    $this->add_group_control(
      \Elementor\Group_Control_Typography::get_type(),
      [
        'name'     => 'filter_btn_typography',
        'selector' => '{{WRAPPER}} .filter-button-group button',
      ]
    );

    // Normal Style
    $this->start_controls_tabs('filter_btn_tabs');

    $this->start_controls_tab(
      'filter_btn_normal',
      [
        'label' => esc_html__('Normal', 'tjcore'),
      ]
    );

    $this->add_control(
      'filter_btn_text_color',
      [
        'label'     => esc_html__('Text Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .filter-button-group button' => 'color: {{VALUE}};',
        ],
      ]
    );

    $this->add_control(
      'filter_btn_bg_color',
      [
        'label'     => esc_html__('Background Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .filter-button-group button' => 'background: {{VALUE}};',
        ],
      ]
    );

    $this->end_controls_tab();

    // Hover/Active Style
    $this->start_controls_tab(
      'filter_btn_hover',
      [
        'label' => esc_html__('Hover / Active', 'tjcore'),
      ]
    );

    $this->add_control(
      'filter_btn_hover_text_color',
      [
        'label'     => esc_html__('Text Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .filter-button-group button:hover, {{WRAPPER}} .filter-button-group button.active' => 'color: {{VALUE}};',
        ],
      ]
    );

    $this->add_control(
      'filter_btn_hover_bg_color',
      [
        'label'     => esc_html__('Background Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .filter-button-group button:hover, {{WRAPPER}} .filter-button-group button.active' => 'background: {{VALUE}};',
        ],
      ]
    );

    $this->end_controls_tab();

    $this->end_controls_tabs();

    // Border
    $this->add_group_control(
      \Elementor\Group_Control_Border::get_type(),
      [
        'name'     => 'filter_btn_border',
        'selector' => '{{WRAPPER}} .filter-button-group button',
      ]
    );

    // Border radius
    $this->add_control(
      'filter_btn_border_radius',
      [
        'label'      => esc_html__('Border Radius', 'tjcore'),
        'type'       => Controls_Manager::DIMENSIONS,
        'selectors'  => [
          '{{WRAPPER}} .filter-button-group button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
      ]
    );

    // Padding
    $this->add_responsive_control(
      'filter_btn_padding',
      [
        'label'      => esc_html__('Padding', 'tjcore'),
        'type'       => Controls_Manager::DIMENSIONS,
        'size_units' => ['px', '%', 'em'],
        'selectors'  => [
          '{{WRAPPER}} .filter-button-group button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
      ]
    );

    $this->end_controls_section();

    $this->start_controls_section(
      'section_style_service',
      [
        'label' => esc_html__('Wrapper style', 'tjcore'),
        'tab'   => Controls_Manager::TAB_STYLE,
      ]
    );
    $this->add_group_control(
      \Elementor\Group_Control_Background::get_type(),
      [
        'name'     => 'tj_wrapper_background',
        'exclude' => ['image', 'video'],
        'selector' => '{{WRAPPER}} .project-item',
      ]
    );

    $this->add_control(
      'wrapper_overlay_color',
      [
        'label'     => __('Overlay Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .project-item::before' => 'background: {{VALUE}}',
        ],
      ]
    );
    $this->add_control(
      'wrapper_overlay_opacity',
      [
        'label'     => __('Opacity', 'tjcore'),
        'type'      => Controls_Manager::NUMBER,
        'selectors' => [
          '{{WRAPPER}} .project-item::before' => 'opacity: {{VALUE}}',
        ],
      ]
    );

    $this->add_group_control(
      \Elementor\Group_Control_Border::get_type(),
      [
        'name' => 'wrapper_border',
        'selector' => '{{WRAPPER}} .project-item',
      ]
    );
    $this->add_responsive_control(
      'tj_wrapper_padding',
      [
        'label'      => esc_html__('Padding', 'tjcore'),
        'type'       => Controls_Manager::DIMENSIONS,
        'size_units' => ['px', '%', 'em', 'rem'],
        'selectors'  => [
          '{{WRAPPER}} .project-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
      ]
    );
    $this->add_responsive_control(
      'tj_wrapper_margin',
      [
        'label'      => esc_html__('Margin', 'tjcore'),
        'type'       => Controls_Manager::DIMENSIONS,
        'size_units' => ['px', '%', 'em', 'rem'],
        'selectors'  => [
          '{{WRAPPER}} .project-item' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
      ]
    );
    $this->end_controls_section();


    // title
    $this->start_controls_section(
      '_style_stitle',
      [
        'label' => esc_html__('Title', 'tjcore'),
        'tab'   => Controls_Manager::TAB_STYLE,
      ]
    );
    $this->add_group_control(
      \Elementor\Group_Control_Typography::get_type(),
      [
        'name'     => 'tj_stitle_typography',
        'selector' => '{{WRAPPER}} .project-item .title',
      ]
    );
    $this->start_controls_tabs('stitle_style_tabs');
    $this->start_controls_tab(
      'stitle_normal_tab',
      [
        'label' => esc_html__('Normal', 'tjcore'),
      ]
    );
    $this->add_control(
      'tj_stitle_color',
      [
        'label'     => __('Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .project-item .title' => 'color: {{VALUE}}',
        ],
      ]
    );
    $this->end_controls_tab();
    $this->start_controls_tab(
      'style_stitle_hover_tab',
      [
        'label' => esc_html__('Hover', 'tjcore'),
      ]
    );
    $this->add_control(
      'tj_stitle_hcolor',
      [
        'label'     => __('Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .project-item .title:hover a' => 'color: {{VALUE}}',
        ],
      ]
    );
    $this->end_controls_tab();
    $this->end_controls_tabs();

    $this->add_responsive_control(
      'tj_stitle_padding',
      [
        'label'      => esc_html__('Padding', 'tjcore'),
        'type'       => Controls_Manager::DIMENSIONS,
        'size_units' => ['px', '%', 'em', 'rem'],
        'selectors'  => [
          '{{WRAPPER}} .project-item .title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
      ]
    );
    $this->add_responsive_control(
      'tj_stitle_margin',
      [
        'label'      => esc_html__('Margin', 'tjcore'),
        'type'       => Controls_Manager::DIMENSIONS,
        'size_units' => ['px', '%', 'em', 'rem'],
        'selectors'  => [
          '{{WRAPPER}} .project-item .title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        ],
      ]
    );
    $this->end_controls_section();

    $this->start_controls_section(
      '_style_stext_category',
      [
        'label' => esc_html__('Category', 'tjcore'),
        'tab'   => Controls_Manager::TAB_STYLE,
      ]
    );
    $this->add_group_control(
      \Elementor\Group_Control_Typography::get_type(),
      [
        'name'     => 'tj_stext_btn_typography',
        'selectors' => '{{WRAPPER}} .project-item .project-content .tag-list a',
      ]
    );
    $this->start_controls_tabs('stext_btn_style_tabs');
    $this->start_controls_tab(
      'stext_cat_normal_tab',
      [
        'label' => esc_html__('Normal', 'tjcore'),
      ]
    );
    $this->add_control(
      'tj_stext_cat_color',
      [
        'label'     => __('Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .project-item .project-content .tag-list a' => 'color: {{VALUE}}',
        ],
      ]
    );

    $this->add_control(
      'tj_stext_cat_bgcolor',
      [
        'label'     => __('Background Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .project-item .project-content .tag-list a' => 'background: {{VALUE}}',
        ],
      ]
    );
    $this->end_controls_tab();
    $this->start_controls_tab(
      'style_stext_cat_hover_tab',
      [
        'label' => esc_html__('Hover', 'tjcore'),
      ]
    );
    $this->add_control(
      'tj_stext_cat_hcolor',
      [
        'label'     => __('Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .project-item .project-content .tag-list a:hover' => 'color: {{VALUE}}',
        ],
      ]
    );
    $this->add_control(
      'tj_stext_cat_hbgcolor',
      [
        'label'     => __('Background Color', 'tjcore'),
        'type'      => Controls_Manager::COLOR,
        'selectors' => [
          '{{WRAPPER}} .project-item .project-content .tag-list a:hover' => 'background: {{VALUE}}',
        ],
      ]
    );
    $this->end_controls_tab();
    $this->end_controls_tabs();
    $this->end_controls_section();
  }

  /**
   * Render the widget on the frontend
   */
  protected function render()
  {

    $settings = $this->get_settings_for_display();
    extract($settings);
    $posts_per_page = ! empty($settings['posts_per_page']) ? intval($settings['posts_per_page']) : 6;
    $categories     = ! empty($settings['category']) ? $settings['category'] : [];
    $title_limit    = ! empty($settings['tj_post_title_word']) ? intval($settings['tj_post_title_word']) : 6;

    $paged = get_query_var('paged') ?: (get_query_var('page') ?: 1);

    $args = [
      'paged'          => $paged,
      'post_type'      => 'projects',
      'post_status'    => 'publish',
      'posts_per_page' => $posts_per_page,
    ];

    if (! empty($categories)) {
      $args['tax_query'] = [
        [
          'taxonomy' => 'projects-cat',
          'field'    => 'slug',
          'terms'    => $categories,
        ],
      ];
    }

    $query = new \WP_Query($args);
?>

    <div class="tj-project-filter">
      <div class="portfolio-menu filter-button-group">
        <?php if (! empty($categories)) : ?>
          <?php if (! empty($settings['tj_all_btn'])) : ?>
            <button class="active" data-filter="*">
              <?php echo esc_html($settings['tj_all_btn']); ?>
            </button>
          <?php endif; ?>

          <?php foreach ($categories as $term_slug) :
            $term = get_term_by('slug', $term_slug, 'projects-cat');
            if (! $term || is_wp_error($term)) {
              continue;
            }
          ?>
            <button data-filter=".<?php echo esc_attr($term_slug); ?>">
              <?php echo esc_html($term->name); ?>
            </button>
          <?php endforeach; ?>
        <?php endif; ?>
      </div>

      <div class="row grid">
        <?php if ($query->have_posts()) : ?>
          <?php while ($query->have_posts()) : $query->the_post(); ?>

            <?php
            $item_terms = get_the_terms(get_the_ID(), 'projects-cat');
            $class_names = '';

            if ($item_terms && ! is_wp_error($item_terms)) {
              foreach ($item_terms as $item_term) {
                $class_names .= ' ' . sanitize_html_class($item_term->slug);
              }
            }
            ?>
            <div class="col-xl-<?php echo esc_attr($desktop_column); ?> col-lg-<?php echo esc_attr($laptop_column); ?> col-md-<?php echo esc_attr($tablet_column); ?> col-<?php echo esc_attr($mobile_column); ?> <?php echo esc_attr($class_names); ?>">
              <div class="project-item">
                <?php if (has_post_thumbnail()): ?>
                  <div class="project-image">
                    <a class="cursor-hide" href="<?php the_permalink(); ?>">
                      <?php the_post_thumbnail('full'); ?>
                    </a>
                  </div>
                <?php endif; ?>
                <div class="project-content">
                  <div class="project-title">
                    <h4 class="title">
                      <a href="<?php the_permalink(); ?>">
                        <?php echo wp_trim_words(get_the_title(), $title_limit, '...'); ?>
                        <i class="tji-arrow-right"></i>
                      </a>
                    </h4>
                  </div>

                  <?php if (!empty($item_terms) && !empty($settings['tj_category'])): ?>
                    <div class="tag-list">
                      <?php
                      $category_names = array();
                      foreach ($item_terms as $cat) {
                        $category_names[] = '<a href="' . esc_url(get_category_link($cat->term_id)) . '">' . esc_html($cat->name) . '</a>';
                      }
                      echo implode(' ', $category_names);
                      ?>
                    </div>
                  <?php endif; ?>
                </div>
              </div>
            </div>
          <?php endwhile;
          wp_reset_postdata(); ?>
        <?php endif; ?>
      </div>
    </div>

<?php
  }

  protected function content_template() {}
}

$widgets_manager->register(new TJ_Project_Filter());
